package jr.ccloader.devices;

/**
 * The programming class for the c-control 
 * 
 * It allows programming the c-control units with native code.
 * The Code was partly ported from the C# Code from Thomas Beier.
 * It saved me a lot of time to implement the loader in java
 * 
 * Jens Riebold, jens.riebold AT web.de
 * 
 */

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import jr.ccloader.comm.ComControl;
import jr.ccloader.util.WorkQueue;

public abstract class Device
{
	public static final int RESET_1 = 0;
	public static final int RESET_2 = 1;
	public static final int LOADING = 2;
	
	private ComControl com = null;
	private WorkQueue<ComMonitor> writequeue = null;
	private WorkQueue<ComMonitor> triggerqueue = null;
	private WorkQueue<ComMonitor> queue = null;
	private boolean recv_running = false;
	private boolean send_running = false;
	private ByteArrayOutputStream stream = null;
	private byte[]	programBuffer = null;
	
	private class ComMonitor
	{
		final public int count;
		public byte[] buffer;
		public boolean valid;
		
		public ComMonitor(int count, byte[] buffer)
		{
			this.count = count;
			this.buffer = buffer;
			this.valid = true;
		}
		
		public void setInvalid()
		{
			this.valid = false;
		}
		
		public String toString()
		{
			StringBuffer buf = new StringBuffer();
			buf.append("ComMonitor: count=").append(count).append(" valid=").append(valid).append(" buffer=");			
			if (buffer == null)
				buf.append("null");
			else
				buf.append(hexline(buffer, 0, buffer.length));
			return buf.toString();
		}
	};
	
	public Device() throws Exception
	{
		com = ComControl.getInstance();
		writequeue = new WorkQueue<ComMonitor>();
		queue = new WorkQueue<ComMonitor>();
		triggerqueue = new WorkQueue<ComMonitor>();
		
		stream = new ByteArrayOutputStream();
		receiveThread();
		sendThread();
	}
	
	public abstract int getMaxAddress();
	abstract int getBlockSize();
	abstract byte[][] getRequestSequence(int type);
	abstract byte[][] getResponseSequence(int type);
	abstract byte[] getCommit();
	abstract byte[] getCommitFinal();
	abstract long[] getSleep(int type);
	abstract boolean[] hasfooterbytes(int type);
	abstract byte[] getFooterbytes();
	
	public byte[] getAndFlushStream()
	{
		synchronized(stream)
		{
			byte[] buf = stream.toByteArray();
			stream.reset();
			System.out.println("getAndFlushStream: " + hexline(buf, 0, buf.length));
			return buf;
		}
	}
	
	public void sleep(long time)
	{
		try
		{
			Thread.sleep(time);
		}
		catch (InterruptedException e)
		{
			
		}
	}
	
	public void sendThread()
	{
		Thread t = new Thread(new Runnable()
		{
			private OutputStream out = null;
			
			public void run()
			{
				try
				{
					send_running = true;
					out = com.getOutputStream();
					while(true)
					{
						ComMonitor monitor = null;
						try
						{
							monitor = writequeue.getWork(0);
							System.out.println("sendThread: monitor=" + monitor.toString());
							if (monitor.valid == false)
								continue;
							
							out.write(monitor.buffer);
						}
						catch (IOException e)
						{
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
						finally
						{
							if (monitor != null)
							{
								System.out.println("sendThread: triggerqueue monitor=" + monitor.toString());							
								triggerqueue.addWork(monitor);
							}
						}
					}
				}
				finally
				{
					send_running = false;
				}
			}			
		});
		t.start();
	}
	
	public void receiveThread()
	{
		Thread t = new Thread(new Runnable()
		{
			public InputStream in = null;
			public byte[] buf = null;
			
			public void readwithfooter(ComMonitor monitor) throws IOException
			{
				System.out.println("enter readwithfooter monitor = " + monitor.toString());
				int n;
				int workerstatus = 0;
				while (recv_running)
				{
					n = in.read(buf);
					for (int i = 0; i<n; i++)
					{
						byte c = buf[i];
						synchronized(stream)
						{
							stream.write(c);
						}
						switch (workerstatus)
						{
						case 0:
							workerstatus = (c==0x10)?1:0;
							break;
						case 1:
							workerstatus = (c==((byte)0xff))?2:0;
							break;
						case 2:
							if (c==0x20)
							{
								System.out.println("leave readwithfooter");
								return;
							}
							else
								workerstatus = 0;
						}
					}
				}				
			}
			
			private void readfixedlength(ComMonitor monitor) throws IOException
			{
				int count = 0;
				System.out.println("enter readfixedlength monitor = " + monitor.toString());
				while (recv_running)
				{
					int n = in.read(buf);
					for (int i = 0; i<n; i++)
					{
						count+=1;
						byte c = buf[i];
						synchronized(stream)
						{
							stream.write(c);
						}
						if (count == monitor.count)
						{
							System.out.println("leave readfixedlength");							
							return;
						}
					}
				}				
			}

			
			@Override
			public void run()
			{
				recv_running = true;
				in = com.getInputStream();
				buf = new byte[1024];
				int n;
				try
				{
					// First flush all data in the Buffer
					while ((n = in.available())>0)
					{
						in.read(buf, 0, (n<buf.length)?n:buf.length);
					}
					
					while (recv_running == true)
					{
						ComMonitor monitor = triggerqueue.getWork(0);
						System.out.println("readthread: dequeue monitor: " + monitor.toString());
						if (monitor.valid == false)
							continue;
						
						getAndFlushStream();
						System.out.println("CControlReader: before cirtical section");
						if (monitor.count <= 0)
						{
							readwithfooter(monitor);
						}
						else
						{
							readfixedlength(monitor);
						}
						monitor.buffer = getAndFlushStream();
						System.out.println("readthread: queue result monitor: " + monitor.toString());						
						queue.addWork(monitor);
					}
				}
				catch (IOException e)
				{
					System.out.println("receiveThread: IOException e=" + e.getMessage());
				}
				finally
				{
					recv_running = false;
				}
				return;
			}
		}, "CControlReader");
		t.start();
	}
	
	public boolean checkStream(ComMonitor m, byte[] vgl)
	{
		System.out.println("checkStream: " + m.toString());
		final byte[] footer = getFooterbytes();
		String s2 = "read: " + Device.hexline(m.buffer, 0, m.buffer.length);
		System.out.println(s2);
		if (vgl == null)
		{
			String s1 = "vgl: nothing to compare";
			System.out.println(s1);
			return true;
		}
		String s1 = "vgl:  " + Device.hexline(vgl, 0, vgl.length);
		System.out.println(s1);
		if (m.buffer.length != ((m.count>0)?vgl.length:vgl.length+footer.length))
			return false;
		
		for (int i = 0; i<m.buffer.length; i++)
		{
			int pos2 = i-vgl.length;
			if (pos2 < 0)
			{
				if (m.buffer[i] != vgl[i])
					return false;
			}
			else
			{
				if (m.buffer[i] != footer[pos2])
					return false;				
			}
		}
		return true;
	}
	
	public boolean reset()
	{
		if (startSequence(RESET_1))
			return true;
		
		return startSequence(RESET_2);
	}
	
	public boolean init()
	{
		return startSequence(LOADING);
	}
	
	public boolean startSequence(int type)
	{
		byte[][] init = getRequestSequence(type);
		byte[][] response = getResponseSequence(type);
		long[] sleep = getSleep(type);
		boolean[] hasfooter = hasfooterbytes(type);

		for (int i=0; i<init.length; i++)
		{
			ComMonitor writeMon = new ComMonitor(hasfooter[i]?0:response[i].length, init[i]);
			writequeue.addWork(writeMon);
			System.out.println("send: " + Device.hexline(init[i], 0, init[i].length));
			sleep(sleep[i]);
			ComMonitor m = queue.getWork(3000);
			if (m == null)
			{
				writeMon.setInvalid();
				System.out.println("TimeOut at initalize");
				byte[] buf = getAndFlushStream();
				String s1 = "Received:  " + Device.hexline(buf, 0, buf.length);
				System.out.println(s1);					
				return false;
			}
			
			if (checkStream(m, response[i]) == false)
			{
				System.out.println(i + ".squence failed");
				return false;
			}
			else
			{
				System.out.println(i + ".squence successful");					
			}
		}
		return true;
	}
	
    // send the hexfile to the MC
    public boolean writeHex()
    {
    	try
    	{
	        // a buffer to store all bytes of a package before sending
	        // A List is used because there are some byte replacments: 
	        // (0x00 -> 0x10 0x01)
	        // (0x0D -> 0x10 0x02) 
	        // (0x10 -> 0x10 0x10)
	        // so the actual package size is >= 256 (last package can be less)
	    	ByteArrayOutputStream transmit = new ByteArrayOutputStream();
	
	        int ct_package = 0;
	        int ct_byte = 0;
	
	        // needed for package numbering (e.g WF0003 for package 3)
	        byte w = 'W';
	        byte f = 'F';
	
	        // second half (last 4 byte) of the final checksum (first half is the byte counter)
	        // ist just a sum of all byte values modulo 2^16
	        int cs = 0;
	
	        // send data
	        for (ct_byte = 0; ct_byte < programBuffer.length; ct_byte++)
	        {
	            // write in packages of 256 byte
	            if (ct_byte % getBlockSize() == 0)
	            {
	                // writing end of package
	                if (ct_byte != 0)
	                {
	                    transmit.write(0x0D); // package ending
	                    // write data and check if data was transmitted correctly
	                    if (!preparePackage(ct_package, transmit))
	                        return false;
	                }
	
	                // starting new package
	                transmit.reset();
	
	                // writing begin of package
	                transmit.write(w);
	                transmit.write(f);
	
	                // Convert the package counter to hexadezimal aasci values (lower case)
	                // e.g.: 001a becomes 0x30 0x30 0x31 0x61
	                transmit.write(intToAasciHex(ct_package, 4));
	
	                ct_package++;
	            }
	
	            // updating checksum
	            
	            cs += (programBuffer[ct_byte] & 0x7F);
	            if ((programBuffer[ct_byte] & 0x80) == 0x80)
	            	cs += 128;
	
	            switch (programBuffer[ct_byte])
	            {
	                case 0x00:
	                	transmit.write(0x10);
	                	transmit.write(0x01);
	                    break;
	                case 0x0D:
	                	transmit.write(0x10);
	                	transmit.write(0x02);
	                    break;
	                case 0x10:
	                	transmit.write(0x10);
	                	transmit.write(0x10);
	                    break;
	                default:
	                	transmit.write(programBuffer[ct_byte]);
	                    break;
	            }
	        }
	
	        // write last package and checksum
	        transmit.write(0x0D);
	        // write data and check if data was transmitted correctly
	        if (preparePackage(ct_package-1, transmit) == false)
	            return false;
	        
	        System.out.println("Now write the checksum");
	        if (writeChecksum(ct_byte, cs) == false)
	        {
	        	System.out.println("checksum failed");
	        	return false;
	        }
	
	        return true;
    	}
    	catch (IOException e)
    	{
    		System.out.println("writehex: IOException e=" + e.getMessage());    		
    	}
    	return false;
    }
    
    private byte[] intToAasciHex(int value, int len)
	{
    	String s = Integer.toHexString(value);
    	for (int i=s.length(); i<len; i++)
    		s = "0" + s;
    	s = s.toLowerCase();
    	System.out.println("intToAasciHex: len" + len + " value=" + value + " (" + s + ")");
    	return s.getBytes();
	}

	private boolean writeChecksum(int ct_byte, int cs) throws IOException
	{
        ByteArrayOutputStream transmit = new ByteArrayOutputStream();
        transmit.write(0x43); 
        transmit.write(0x53); // "CS"

        // Converting the byte counter to aasci
        transmit.write(intToAasciHex(ct_byte, 5));

        // Converting the checksum to aasci
        transmit.write(intToAasciHex(cs, 4));

        // package ending
        transmit.write(0x0D);
        byte[] toSend = transmit.toByteArray();
        // The final
        transmit.write(getCommitFinal());
        byte[] toRecv = transmit.toByteArray();
        
        // write data and check if data was transmitted correctly
        if (!writeAndCheckPackage(-1, toSend, toRecv))
            return false;

        return true;

	}

	// sends the package "send" to MC and checks the response
    boolean preparePackage(int packageNr, ByteArrayOutputStream send) throws IOException
    {
        // sending package
    	byte[] toSend = send.toByteArray();
		send.write(getCommit());
    	byte[] toRecv = send.toByteArray();
    	
    	for (int i=0; i<4; i++)
    	{
    		if (writeAndCheckPackage(packageNr, toSend, toRecv) == true)
    			return true;
    	}
    	return false;
    }

    boolean writeAndCheckPackage(int packageNr, byte[] toSend, byte[] toRecv) throws IOException
    {
		ComMonitor writeMon = new ComMonitor(0, toSend);
		writequeue.addWork(writeMon);
 
 		ComMonitor m = null;
		while (true)
		{
			m = queue.getWork(3000);
			if (m == null)
				break;
			if (m.valid)
				break;
		}
		if (m == null)
		{
			writeMon.setInvalid();
			System.out.println("TimeOut at sending package: packageNr=" + packageNr);
			byte[] buf = getAndFlushStream();
			String s1 = "Received:  " + Device.hexline(buf, 0, buf.length);
			System.out.println(s1);					
			return false;
		}
		
		if (checkStream(m, toRecv) == false)
		{
			System.out.println(packageNr + ".package failed");
			return false;
		}
		else
		{
			System.out.println(packageNr + ".package successful");					
		}
        return true;    	
    }
    
	public static String hexdump(byte[] buf, int count)
	{
		StringBuffer out = new StringBuffer();
		for (int i=0; i<buf.length; i+=count)
		{
			out.append(hexline(buf, i, count)).append("\n");
		}
		return out.toString();
	}
	
	public static String hexline(byte[] buf, int pos, int count)
	{
		StringBuffer out = new StringBuffer();
		String t = Integer.toString(pos & 0xfffff, 16).toUpperCase();
		while (t.length() < 5)
			t = "0" + t;
		out.append(t).append(": ");
		for (int i=0; ((i<count) && ((pos+i) < buf.length)); i++)
		{
			t = Integer.toString(buf[i+pos] & 0xff, 16).toUpperCase();
			if (t.length() == 1) t = "0" + t;
			out.append(t).append(" ");
		}
		return out.toString();
	}
	public void close()
	{
		try	{ com.getInputStream().close();} catch (IOException e) { }
		try	{ com.getOutputStream().close();} catch (IOException e) { }
	}

	public void setProgramBuffer(byte[] buffer)
	{
		this.programBuffer = buffer;
	}

}
