/******************************************************************/
/* Treiberbibliothek zum CCTools-I2C-COM  (www.cctools.eu)        */
/*  fr C-Control Pro
/*                                                                */
/* Autor     : Andr Helbig (www.cc2net.de)                       */
/* Vorlage   : Andr Helbig (C-Control II, i2ccom.c2)             */
/* Versionen : 1.0                                                */
/* Datum     : 31. Januar 2010                                    */
/* Getestet  : ja                                                 */
/******************************************************************/

byte COM[31], FIFOcnt[31];


//--- Baudraten ----------------------------------------------------------------
#define baud300  1536
#define baud600   768
#define baud1200  384
#define baud2400  192
#define baud4800   96
#define baud7200   64
#define baud9600   48
#define baud14400  32
#define baud19200  24
#define baud28800  16
#define baud38400  12
#define baud57600   8
#define baud115200  4
#define baud230400  2
#define baud460800  1

//--- Datenformate -------------------------------------------------------------
// 8Bit, 1 Stopbit
#define set8N1     0x03 // 0b00000011
#define set8E1     0b00011011
#define set8O1     0b00001011
// 8Bit, 2 Stopbits
#define set8N2     0b00000111
#define set8E2     0b00011111
#define set8O2     0b00001111
// 7Bit, 1 Stopbit
#define set7N1     0b00000010
#define set7E1     0b00011010
#define set7O1     0b00001010
// 7Bit, 2 Stopbits
#define set7N2     0b00000110
#define set7E2     0b00011110
#define set7O2     0b00001110
// 6Bit, 1 Stopbit
#define set6N1     0b00000001
#define set6E1     0b00011001
#define set6O1     0b00001001
// 6Bit, 2 Stopbits
#define set6N2     0b00000101
#define set6E2     0b00011101
#define set6O2     0b00001101
// 5Bit, 1 Stopbit
#define set5N1     0b00000000
#define set5E1     0b00011000
#define set5O1     0b00001000
// 5Bit, 1,5 Stopbits
#define set5N2     0b00000100
#define set5E2     0b00011100
#define set5O2     0b00001100
// Allgemein:(Verknpfung mit "or")
#define set8Bit    0b00000011 // 8 Datenbits
#define set7Bit    0b00000010 // 7 Datenbits
#define set6Bit    0b00000001 // 6 Datenbits
#define set5Bit    0b00000000 // 5 Datenbits
#define set1Stop   0b00000000 // 1 Stopbit
#define set2Stop   0b00000100 // 2 Stopbits (1,5 bei 5 Datenbits)
#define setP_no    0b00000000 //No Parity
#define setP_even  0b00011000 //Even Parity
#define setP_odd   0b00001000 //Odd Parity
#define setP_stick 0b00100000 //Stick Parity
//------------------------------------------------------------------------------

#define FifoWR 0xF7
#define FifoRD 0xEF

#define IER 1 //Interrupt-Enable-Register
#define IIR 2 //Interrupt Ident.Reg. (readonly)
#define FCR 2 //FIFO Control Reg. (writeonly)
#define LCR 3 //LineControl-Register
#define MCR 4 //Modem-Control-Register
#define LSR 5 //Line Status-Register
#define MSR 6 //Modem-Status-Register
#define SCR 7 //Scratch Register (free register)

//------------------------------------------------------------------------------
//-- SYSTEM-Funktionen ---------------------------------------------------------
//------------------------------------------------------------------------------
byte i2ccom_getsub(byte comport)
{
 return (COM[comport] & 1) << 5;
}

void i2ccom_wrdatapcf(byte data)
{
 I2C_Start();
 I2C_Write(COM[0]);
 I2C_Write(data);
 I2C_Stop();
}

void i2ccom_wrpulse(byte addr, byte reg)
{
 I2C_Start();
 I2C_Write(addr);
 I2C_Write(reg);
 I2C_Write(reg & FifoWR);
 I2C_Write(reg);
 I2C_Stop();
}


//------------------------------------------------------------------------------
//-- Funktionen ----------------------------------------------------------------
//------------------------------------------------------------------------------

//--- Grundeinstellungen -------------------------------------------------------

/*****************************************/
/* Daten-PCF (0-7 PCF8574, 8-15 PCF8574A)*/
/*****************************************/
void i2ccom_setDataPCF(byte device)
{
 if(device > 7)
     COM[0]=(device+48)*2;
   else
     COM[0]= device*2+64;
}

/*****************************************/
/* Schnittstelle definieren (COM 1 - 30) */
/* Ctrl=0-7:PCF8574,8-15:PCf8574A        */
/* sub= Jumper Sub:false="1-2" true="2-3"*/
/*****************************************/
void i2ccom_defineCOM(byte com, byte Ctrl, byte sub)
{
 if(com==0 || com > 30 || Ctrl > 15) return;
 if(Ctrl < 8)
   COM[com]=((Ctrl << 1)+64) | (sub!=0 & 1);
  else
   COM[com]=((Ctrl+48) << 1) | (sub!=0 & 1);
}

/**********************************/
/* Schnittstelle Initialisieren   */
/* Default: 8N1, RXD interrupt    */
/**********************************/
int i2ccom_initCOM(byte Port, byte autoflow)
{
 i2ccom_setReg(Port, SCR, 0xAA);
 if(i2ccom_getReg(Port, SCR)!=0xAA) return 0;
 i2ccom_setReg(Port, SCR, 0x55);
 if(i2ccom_getReg(Port, SCR)!=0x55) return 0;
 i2ccom_setReg(Port, FCR, 0x09);
 i2ccom_setReg(Port, LCR, set8N1|0x80);
 i2ccom_setReg(Port, FCR, 0x29);
 i2ccom_setReg(Port, LCR, set8N1);
 i2ccom_setRate(Port, 48);
 autoflow=(autoflow!=0) & 0x22;
 i2ccom_setReg(Port, MCR, autoflow);
 i2ccom_setReg(Port, IER, 0x01);
 return -1;
}

/**********************************/
/* Daten-Format einstellen        */
/**********************************/
void i2ccom_setFormat(byte Port,byte format)
{
 i2ccom_setReg(Port, LCR, format);
}

/**********************************/
/* Baudrate einstellen            */
/* divisor=460800 / Baudrate      */
/**********************************/
void i2ccom_setRate(byte Port, int divisor)
{byte LCRreg, reg;
 reg = 0x58 | i2ccom_getsub(Port);

 LCRreg = i2ccom_getReg(Port, LCR);
 // set DLAB-Bit
 i2ccom_wrdatapcf(LCRreg | 0x80);
 i2ccom_wrpulse(COM[Port] & 0xFE, LCR | reg);

 // write Divisor-Latch LSB
 i2ccom_wrdatapcf(divisor);
 i2ccom_wrpulse(COM[Port] & 0xFE, reg);

 // write Divisor-Latch LSB
 i2ccom_wrdatapcf(divisor >> 8);
 i2ccom_wrpulse(COM[Port] & 0xFE, reg | 1);

 // clear DLAB-Bit
 i2ccom_wrdatapcf(LCRreg & 0x7F);
 i2ccom_wrpulse(COM[Port] & 0xFE, LCR | reg);

 i2ccom_wrdatapcf(0xFF);
}



//--- Control / Status ---------------------------------------------------------

/**********************************/
/* Interrupt-Register lesen       */
/**********************************/
byte i2ccom_getIIR(byte Port)
{
 return i2ccom_getReg(Port,IIR) ^ 1;
}

/**********************************/
/* Interrupt-En.-Register setzen  */
/**********************************/
void i2ccom_setIER(byte Port, byte Data)
{
 i2ccom_setReg(Port, IER, Data);
}

/**********************************/
/* Interrupt-En.-Register lesen  */
/**********************************/
byte i2ccom_getIER(byte Port)
{
 return i2ccom_getReg(Port,IER);
}

/**********************************/
/* Line-Status-Register lesen      */
/**********************************/
byte i2ccom_getLSR(byte Port)
{
 return i2ccom_getReg(Port,LSR);
}

/**********************************/
/* Modem-Status-Register lesen    */
/**********************************/
byte i2ccom_getMSR(byte Port)
{
 return i2ccom_getReg(Port,MSR);
}

/**********************************/
/* Register lesen                 */
/**********************************/
byte i2ccom_getReg(byte Port, byte reg)
{byte data;
 reg = reg | 0x58 | i2ccom_getsub(Port);
 I2C_Start();
 I2C_Write(COM[Port] & 0xFE);
 I2C_Write(reg);
 I2C_Write(reg & FifoRD);
 I2C_Stop();
 I2C_Start();
 I2C_Write(COM[0] | 1);
 data = I2C_Read_NACK();
 I2C_Stop();
 I2C_Start();
 I2C_Write(COM[Port] & 0xFE);
 I2C_Write(reg);
 I2C_Stop();
 return data;
}

/**********************************/
/* Register setzen                */
/**********************************/
void i2ccom_setReg(byte Port, byte reg, byte data)
{
 reg = reg | 0x58 | i2ccom_getsub(Port);
 //Msg_WriteInt(reg);
 i2ccom_wrdatapcf(data);
 i2ccom_wrpulse(COM[Port] & 0xFE, reg);
 i2ccom_wrdatapcf(0xFF);
}

/**********************************/
/* DTR setzen                     */
/**********************************/
void i2ccom_DTRenable(byte Port, byte state)
{
 state = (state!=0) & 1;
 state = (i2ccom_getReg(Port, MCR) & 0xFE) | state;
 i2ccom_setReg(Port, MCR, state);
}

/**********************************/
/* RTS setzen                     */
/**********************************/
void i2ccom_RTSenable(byte Port, byte state)
{
 state = (state!=0) & 0x02;
 state = i2ccom_getReg(Port, MCR) | state;
 i2ccom_setReg(Port, MCR, state);
}

/**********************************/
/* Auto-Flow setzen               */
/* (automatisches Handshake)      */
/**********************************/
void i2ccom_AFenable(byte Port, byte state)
{
 state = (state!=0) & 0x20;
 state = i2ccom_getReg(Port, MCR) | state;
 i2ccom_setReg(Port, MCR, state);
}

/**********************************/
/* CTS abfragen                   */
/**********************************/
int i2ccom_getCTS(byte Port)
{
 return (i2ccom_getReg(Port, MSR) & 0x10)!=0;
}

/**********************************/
/* DSR abfragen                   */
/**********************************/
int i2ccom_getDSR(byte Port)
{
 return (i2ccom_getReg(Port, MSR) & 0x20)!=0;
}

/**********************************/
/* DCD abfragen                   */
/**********************************/
int i2ccom_getDCD(byte Port)
{
 return (i2ccom_getReg(Port, MSR) & 0x80)!=0;
}

/**********************************/
/* Ring-Indicator abfragen        */
/**********************************/
int i2ccom_getRI(byte Port)
{
 return (i2ccom_getReg(Port, MSR) & 0x40)!=0;
}

/**********************************/
/* TERI abfragen                  */
/**********************************/
int i2ccom_getTERI(byte Port)
{
 return (i2ccom_getReg(Port, MSR) & 0x04)!=0;
}


//--- Output ---------------------------------------

/**********************************/
/* Sendebereitschaft prfen       */
/**********************************/
int i2ccom_ready(byte Port)
{
 return (i2ccom_getReg(Port, LSR) & 0x60)!=0;
}

/**********************************/
/* Einzelnes Zeichen senden       */
/**********************************/
void i2ccom_put(byte Port,byte Data)
{byte reg;
 reg = 0x58 | i2ccom_getsub(Port);
 FIFOcnt[Port]=FIFOcnt[Port]+1;
 if(FIFOcnt[Port] & 0x40)
   {while((i2ccom_getReg(Port, LSR) & 0x60)==0) {}
    FIFOcnt[Port]=1;}
 i2ccom_wrdatapcf(Data);
 i2ccom_wrpulse(COM[Port] & 0xFE, reg);

 i2ccom_wrdatapcf(0xFF);
}


/**********************************/
/* String senden                  */
/**********************************/
/*void print(byte Port, string s)
{
}

/**********************************/
/* Byte-Array senden              */
/**********************************/
void i2ccom_send(byte Port, byte buf[], int len)
{int i;
 byte reg;
 reg = 0x58 | i2ccom_getsub(Port);
 for(i=0;i<len;i++)
 {
  FIFOcnt[Port]=FIFOcnt[Port]+1;
  if(FIFOcnt[Port] & 0x40)
    {while((i2ccom_getReg(Port, LSR) & 0x60)==0) {}
     FIFOcnt[Port]=1;}
  i2ccom_wrdatapcf(buf[i]);
  i2ccom_wrpulse(COM[Port] & 0xFE, reg);
 }
 i2ccom_wrdatapcf(0xFF);
}

/**********************************/
/* CR+LF senden                   */
/**********************************/
void i2ccom_ret(byte Port)
{
 i2ccom_put(Port,13);
 i2ccom_put(Port,10);
}

/**********************************/
/* Tabulator senden               */
/**********************************/
void i2ccom_tab(byte Port)
{
 i2ccom_put(Port,9);
}

/**********************************/
/* Clear Screen senden            */
/* (fr Terminal-Programme)       */
/**********************************/
void i2ccom_clr(byte Port)
{
 i2ccom_put(Port,12);
}


//--- Input --------------------------------------------------------------------

/**********************************/
/* Auf Empfang prfen             */
/**********************************/
int i2ccom_rxd(byte Port)
{
 return (i2ccom_getReg(Port, IIR) & 0x04)!=0;
}

/**********************************/
/* Empfangs-FIFO lschen          */
/**********************************/
void i2ccom_flush(byte Port)
{
 i2ccom_setReg(Port, FCR, 0x0B);
}

/**********************************/
/* Ein einzelnes Zeichen lesen    */
/* Es sollte vorzugsweise mit     */
/* receive() gearbeitet werden !  */
/**********************************/
byte i2ccom_get(byte Port)
{byte data, reg;
 while(!i2ccom_rxd(Port))
 {
 }
 reg = 0x58 | i2ccom_getsub(Port);
 I2C_Start();
 I2C_Write(COM[Port] & 0xFE);
 I2C_Write(reg);
 I2C_Write(reg & FifoRD);
 I2C_Stop();
 I2C_Start();
 I2C_Write(COM[0] | 1);
 data = I2C_Read_NACK();
 I2C_Stop();
 I2C_Start();
 I2C_Write(COM[Port] & 0xFE);
 I2C_Write(reg);
 I2C_Stop();
 return data;
}


//--- Ports OUT 1 & 2 ----------------------------------------------------------
/**********************************/
/* Ports OUT 1 / 2 setzen         */
/* Port=0:OUT1, 1:OUT2            */
/**********************************/
void i2ccom_setPort(byte COMPort, byte Port, byte state)
{
 if(Port>1) return;
 Port=(0x04 << Port);
 if(state) state=-1 & Port;
 Msg_WriteInt(state);Msg_WriteChar(0x20);
 state = ((i2ccom_getReg(COMPort, MCR) & ~Port) | state);
 i2ccom_setReg(COMPort, MCR, state);
}

/**********************************/
/* Ports OUT 1 / 2 abfragen       */
/* Port=0:OUT1, 1:OUT2            */
/**********************************/
int i2ccom_getPort(byte COMPort, byte IO)
{
 if(IO>1) return 0;
 return ((i2ccom_getReg(COMPort, MCR) & (0x04 << IO))!=0);
}

