/*** SensorConditioner.h ***/
//-----------------------------------------------
// Author: Torsten Knorr, create-soft@freenet.de
/*** BUGS ***/
// Maybe you'll find some. Please let me know.
// By the way I am pleased with every kind of feedback.
//-----------------------------------------------
 #ifndef __SENSOR_CONDITIONER__
 #define __SENSOR_CONDITIONER__
//-----------------------------------------------
 #define SENSOR_POLYNOM 0xA005
 #define SENSOR_EEPROM_SIZE 32
 #define SENSOR_CRC_ADDR 29

/*** I2C address ***/
// Hygrosens module default write address
// 0x78 << 1 = 0b1111 000|0 = 0xF0
 #define DEFAULT_WRITE_ADDR (0x78 << 1)
 #define SECOND_WRITE_ADDR1 (0x79 << 1)
 #define SECOND_WRITE_ADDR2 (0x7A << 1)
 #define SECOND_WRITE_ADDR3 (0x7B << 1)
 #define SECOND_WRITE_ADDR4 (0x7C << 1)

/*** Restricted Command Set for Serial Digital Interface ***/
// Commands are always valid

// Start measurement cycle including initialization from EEPROM (350s)
 #define START_CYC_EEP 0x01

// Start measurement cycle including initialization from RAM (220s)
 #define START_CYC_RAM 0x02

// Read data from RAM address 00 .. 0F Writes data from RAM to SIF Output
// Registers Usually followed by Read operation (50s)
// 10 .. 1F
 #define READ_RAM0(a) (0x10 + a)

// Read data from RAM address 10 .. 1F Writes data from RAM to SIF Output
// Register Usually followed by Read operation (50s)
// 20 .. 2F
 #define READ_RAM1(a) (0x20 + a)
// CFGSIF - Configuration of digital serial interface (SIF)
 #define READ_RAM1_CFGSIF 0x27

// Read data from EEPROM address 00 .. 0F Writes data from EEPROM to SIF Output
// Register Usually followed by Read operation (50s)
// 30 .. 3F
 #define READ_EEP0 0x30
// #define READ_EEP0(a) (0x30 + a)

// Read data from EEPROM address 10 .. 1F Writes data from EEPROM to SIF Output
// Register Usually followed by Read operation (50s)
// 40 .. 4F
 #define READ_EEP1(a) (0x40 + a)
 #define READ_EEP1_CFGSIF 0x47

// Configure SIF to Communication Mode OWI (50s)
 #define CFG_SIF_TO_OWI 0x50

// CFG_SIF_TO_SPI Configure SIF to Communication Mode SPI (50s)
// SPI (CKE = 0, CKP = 0)
 #define SET_SIF_2_SPI 0x51
// SPI (CKE = 0, CKP = 1)
 #define SET_SIF_2_SPI_CKP 0x55
// SPI (CKE = 1, CKP = 0)
 #define SET_SIF_2_SPI_CKE 0x59
// SPI (CKE = 1, CKP = 1)
 #define SET_SIF_2_SPI_CKE_CKP 0x5D

// Configure SIF to Communication Mode I2C
 #define CFG_SIF_TO_I2C 0x52

// Set Output DAC to value defined by data bytes [0x0  0x7FF] (100s)
// 2 byte
 #define SET_DAC 0x60

// Start Open Mode In Open Mode the command set is restricted!
// Change to Command Mode is possible (50s)
 #define START_OM 0x70

// Start Normal Operation Mode (NOM) In Normal Operation Mode the command
// set is restricted! Change to Command Mode or Open Mode is NOT possible
// If the ZSC31050 receives a command in NOM, which is not element of
// the reduced command set, this command is ignored, no interrupt for
// the CMC is generated and the measurement cycle is continued.
// If the serial interface works in ICTM mode no acknowledges are generated
// after the command byte.
 #define START_NOM 0x71

// Start Command Mode (CM) Complete command set! Change to Open Mode or
// Normal Operation Mode is possible. The command START_CM is only
// processed if it is received as first command after power-on via digital
// serial interface.
 #define START_CM 0x72

/*** Additional Command Set for Serial Digital Interface ***/
// These commands are only available in Command Mode CM.
// CM is set by command START_CM (0x72hex) as first command after power on.

// Write data to RAM address 00 .. 0F (50s)
// 80 .. 8F 2 Byte
 #define WRITE_RAM0(a) (0x80 + a)

// Write data to RAM address 10 .. 1F (50s)
// 90 .. 9F 2 byte
 #define WRITE_RAM1(a) (0x90 + a)
 #define WRITE_RAM1_CFGSIF 0x97

// Write data to EEPROM address 00 .. 0F (12.5ms / 1)
// A0 .. AF 2 byte
 #define WRITE_EEP0(a) (0xA0 + a)

// Write data to EEPROM address 10 .. 1F (12.5ms / 1)
// B0 .. BF 2 byte
 #define WRITE_EEP1(a) (0xB0 + a)
 #define WRITE_EEP1_CFGSIF 0xB7

// Copy Contents of EEPROM to RAM Restores EEPROM Configuration in RAM (130s)
 #define COPY_EEP2RAM 0xC0

// Copy Contents of RAM to EEPROM Stores RAM Configuration to EEPROM
// (signature is copied without check!!!) (400ms / 32)
 #define COPY_RAM2EEP 0xC3

// Calculates EEPROM Signature and output to SIF Output Register (150s)
 #define GET_EEP_SIGN 0xC8

// GET_EEP_SIGN + write the Signature to EEPROM address 0x1D (12.6ms / 1)
 #define GEN_EEP_SIGN 0xC9

// Calculates RAM Signature and output to SIF Output Register (150s)
 #define GET_RAM_SIGN 0xCA

// GET_RAM_SIGN + write the Signature to RAM address 0x1D (150s)
 #define GEN_RAM_SIGN 0xCB

// Clear EEPROM Sets complete EEPROM to 0x0000 (12.5ms / 1)
 #define CLEAR_EEP 0xCC

// Read ROM version and write it to SIF Output Register
// Usually followed by Read operation (50s)
 #define ROM_VERSION 0xCF


// All D*-Commands: used for calibration process, write raw conversion
// result to SIF Output Registers, do not effect to analog output
// D0, D1 and D2 will return ZADC values of equations in section 2.1.
// D4, D5 and D6 will return ZAZ values of equations in section 2.1.
// D8, D9 and DA will return ZCORR values of equations in section 2.1.

// Start cyclic A/D-conversion at channel Pressure P
 #define START_AD_P 0xD0

//Start cyclic A/D-conversion at channel Temperature T1
 #define START_AD_T1 0xD1

// Start cyclic A/D-conversion at channel Temperature T2
 #define START_AD_T2 0xD2

// Start cyclic A/D-conv at channel Auto-Zero Pressure PAZ
 #define START_AD_PAZ 0xD4

// Start cyclic A/D-conv at channel Auto-Zero Temp TAZ1
 #define START_AD_TAZ1 0xD5

// Start cyclic A/D-conv at channel Auto-Zero TempTAZ2
 #define START_AD_TAZ2 0xD6

// Start cyclic A/D-conv for Pressure P incl. Auto-Zero-Correction
 #define START_AD_P_AZC 0xD8

// Start cyclic A/D-conv for Temperature T1 incl. Auto-Zero-Correction
 #define START_AD_T1_AZC 0xD9

// Start cyclic A/D-conv for Temperature T2 incl. Auto-Zero-Correction
 #define START_AD_T2_AZC 0xDA

// Start cyclic A/D-conversion at channel Common Mode Voltage
// CMV incl. Auto-Zero-Correction (T1AZ)
 #define START_AD_CMV_AZC 0xDB

/*** CFGSIF - Configuration of digital serial interface (SIF) ***/

// Bit 15:9 (default = 0000 000) Alternative ICTM slave address Additional
// slave address for communication via digital serial interface, if
// CFGSIF:SIFID2E = 1. The general slave address is 0x78. If enabled both
// addresses are valid.
 #define SIFID2

// Bit 8 (default = 0) Enable one wire communication during measurement cycle
// Active only if CFGCYC:OWIWIND = 0 (enables start window for one wire
// communication). Makes possible the read-out of the conditioning results
// for pressure and temperature via one wire communication.
 #define OWIE 8

// Bit 7 (default = 0) Disable start window for one wire communication Suppresses
// the 20ms time-slot for starting one wire communication after power-on
// and after restart the measurement cycle by sending the commands
// START_CYC_EEP or START_CYC_RAM
 #define OWIWIND 7

// Bit 6 (default = 0) Enable SIF output of conditioning result temperature 2
// Delivers a continuously updated 15 bit value of temperature 2 for read out
// during measurement cycle via digital serial interface.
 #define SIFOUTT2 6

// Bit 5 (default = 1) Enable SIF output of conditioning result temperature 1
// Delivers a continuously updated 15 bit value of temperature 1 for read out
// during measurement cycle via digital serial interface.
 #define SIFOUTT1 5

// Bit 4 (default = 1) Enable SIF output of conditioning result pressure
// Delivers a continuously updated 15 bit value of measurand for read out during
// measurement cycle via digital serial interface.
 #define SIFOUTP 4

// Bit 3 (default = 0) Enable alternative ICTM slave address
// The alternative address CFGSIF:SIFID2 is only valid for ICTM!
 #define SIFID2E 3

// Bit 2 (default = 1) SPI clock edge select (CPHA)
// 0: sample leading edge 1: sample trailing edge
 #define SPICKE 2

// Bit 1 (default = 0) SPI clock polarity (CPOL)
// 0: SCK low, idle 1: SCK high, idle
 #define SPICKP 1

// Bit 0 (default = 0) SIF mode
// Chose active communication mode:
// 0: ICTM communication 1: SPI communication
 #define SIFMD 0

 #define SIFCFG_DEFAULT 0x0034

/*** Error Codes ***/
// Signature-Check during read out of EEPROM after Power On
// or after SIFCommand COPY_EEP2RAM
 #define ERROR_EEPROM_SIGN 0xCAAA

// Parity-Check at every RAM access
// CFGAPP: SCCD
 #define ERROR_RAM_PARITY 0xCF0F

// Permanent Parity-Check of Configuration Registers
// CFGAPP: SCCD
 #define ERROR_REGISTER_PARITY 0xCE38

// Signature-Check of ROM after Poweron
// CFGCYC: ROMCHE
 #define ERROR_ROM_SIGN 0xCCCC

// Check Arithmetic-Check during measurement cycle
 #define ERROR_ARITHMETIC 0xC1C7

// Watchdog-Check of start-routine and measurement-cycle
 #define ERROR_WATCHDOG 0xC555

// Programmed configuration is not allowed
// (Conflict regarding A/D conversion-time and enabled functionality)
 #define ERROR_CONFIGURATION 0xC333

// Connection-Check of Sensor Bridge
// CFGAPP: SCCD
 #define ERROR_CONNECTION 0xCFCF

// Common Mode Voltage out of limits
// Check if Bridge Common Mode Voltage complies the programmed limits
// CGFCYC: ECMV
// 2b10 + 14bits VIN_CM/2 (cmd: DB)
// VIN_CM  Measured Common Mode Voltage
// TODO
//#define ERROR_VOLTAGE(a) ((a)&(0x8000))

// Reactivation is possible by sending following commands:
// 72, 52, B70034, C9 & 01 via ICTM.
// 0x72 = START_CM
// 0x52 = CFG_SIF_TO_I2C
// 0xB7 = WRITE_EEP1_CFGSIF default 0x0034
// 0xC9 = GEN_EEP_SIGN
// 0x01 = START_CYC_EEP
//------------------------------------------------
 #endif
//------------------------------------------------