/*** Humidity.cc ***/
//------------------------------------------------
// Author: Torsten Knorr, create-soft@freenet.de
/*** BUGS ***/
// Maybe you'll find some. Please let me know.
// By the way I am pleased with every kind of feedback.
//------------------------------------------------
 #include "Humidity.h"
//------------------------------------------------
 float HumidityRead(byte by_write_addr)
    {
    byte aby_data[2];
    byte by_read_addr;
    word w_humidity;
    float f_humidity;

    by_read_addr = by_write_addr + 1;
    aby_data[0] = 0x00;
    aby_data[1] = 0x00;
    if(!I2CReceiveData(by_read_addr, aby_data, 2, false))
        return INVALID_VALUE;

    if(0x80 & aby_data[0])
        return INVALID_VALUE;

    w_humidity = aby_data[0];
    w_humidity = w_humidity << 8;
    w_humidity = w_humidity | aby_data[1];
    f_humidity = w_humidity;
    f_humidity = f_humidity * HUMIDITY_FACTOR;

    if((HUMIDITY_MIN < f_humidity) && (HUMIDITY_MAX > f_humidity))
        return f_humidity;

    return INVALID_VALUE;
    }
//------------------------------------------------
 float HumidityGetTemp(byte by_write_addr)
    {
    byte aby_data[4];
    byte by_read_addr;
    word w_temperature;
    float f_temperature;

    by_read_addr = by_write_addr + 1;
    aby_data[0] = 0x00;
    aby_data[1] = 0x00;
    aby_data[2] = 0x00;
    aby_data[3] = 0x00;
    if(!I2CReceiveData(by_read_addr, aby_data, 4, false))
        return INVALID_VALUE;

    if(0x80 & aby_data[2])
        return INVALID_VALUE;

    w_temperature = aby_data[2];
    w_temperature = w_temperature << 8;
    w_temperature = w_temperature | aby_data[3];
    f_temperature = w_temperature;
    f_temperature = f_temperature * TEMPERATURE_FACTOR;
    f_temperature = f_temperature - TEMPERATURE_OFFSET;

    if((HUMIDITY_MIN_TEMP < f_temperature) && (HUMIDITY_MAX_TEMP > f_temperature))
        return f_temperature;

    return INVALID_VALUE;
    }
//------------------------------------------------
//                  (RH%)   4222.03716 * TC
//      241.2   * ln(---) + ----------------
//                  (100)   241.2 + TC
// dp = ------------------------------------
//                  (RH%)   17.5043    * TC
//      17.5043 - ln(---) - ----------------
//                  (100)   241.2 + TC
//------------------------------------------------
 #if MEGA128 || MEGA128CAN
 float HumidityCalDewPoint(float f_humidity, float f_temperature)
    {
    float f_dew_point;
    float f_ln;
    float f_sum;
    float f_res_1;
    float f_res_2;

    f_ln = ln(f_humidity / 100.0);
    f_sum = 241.2 + f_temperature;

    f_res_1 = 241.2 * f_ln + 4222.03716 * f_temperature / f_sum;
    f_res_2 = 17.5043 - f_ln - 17.5043 * f_temperature / f_sum;

    f_dew_point = f_res_1 / f_res_2;

    return f_dew_point;
    }
//------------------------------------------------
//                  (RH%)   6107.85384 * TC
//      272.186 * ln(---) + ----------------
//                  (100)   272.186 + TC
// fp = ------------------------------------
//                  (RH%)   22.4433    * TC
//      22.4433 - ln(---) - ----------------
//                  (100)   272.186 + TC
//------------------------------------------------
 float HumidityCalFrostPoint(float f_humidity, float f_temperature)
    {
    float f_dew_point;
    float f_ln;
    float f_sum;
    float f_res_1;
    float f_res_2;

    f_ln = ln(f_humidity / 100.0);
    f_sum = 272.186 + f_temperature;

    f_res_1 = 272.186 * f_ln + 6701.85384 * f_temperature / f_sum;
    f_res_2 = 22.4433 - f_ln - 22.4433 * f_temperature / f_sum;

    f_dew_point = f_res_1 / f_res_2;

    return f_dew_point;
    }
//------------------------------------------------
 float HumidityDewPoint(byte by_write_addr)
    {
    float f_humidity;
    float f_temperature;
    float f_dew_point;

    f_humidity = HumidityRead(by_write_addr);
    if(INVALID_VALUE == f_humidity)
        return INVALID_VALUE;

    f_temperature = HumidityGetTemp(by_write_addr);
    if(INVALID_VALUE == f_temperature)
        return INVALID_VALUE;

    if(0 < f_temperature)
        f_dew_point = HumidityCalDewPoint(f_humidity, f_temperature);
    else
        f_dew_point = HumidityCalFrostPoint(f_humidity, f_temperature);

    return f_dew_point;
    }
 #endif
//-----------------------------------------------
 void HumiditySendValues(byte by_write_addr, int i_device_number)
    {
    char ac_msg[40];
    float f_value;

    f_value = HumidityRead(by_write_addr);
    if(INVALID_VALUE != f_value)
        Str_Printf(ac_msg, "%d. Humidity   : %5.2f %c\r", i_device_number, f_value, '%');
    else
        Str_Printf(ac_msg, "%d. Humidity   : ERROR\r", i_device_number);
    Msg_WriteText(ac_msg);

    f_value = HumidityGetTemp(by_write_addr);
    if(INVALID_VALUE != f_value)
        Str_Printf(ac_msg, "%d. Temperature: %5.2f C\r", i_device_number, f_value);
    else
        Str_Printf(ac_msg, "%d. Temperature: ERROR\r", i_device_number);
    Msg_WriteText(ac_msg);

 #if MEGA128 || MEGA128CAN
    f_value = HumidityDewPoint(by_write_addr);
    if(INVALID_VALUE != f_value)
        Str_Printf(ac_msg, "%d. Dew-Point  : %5.2f C\r", i_device_number, f_value);
    else
        Str_Printf(ac_msg, "%d. Dew-Point  : ERROR\r", i_device_number);
    Msg_WriteText(ac_msg);
 #endif
    }
//------------------------------------------------