/*** DCF77Time.cc ***/
//------------------------------------------------
 #include "DCF77Time.h"
//------------------------------------------------
 #ifdef MEGA128
 void DCF77StartTimer3A $asm("TagDCF77StartTimer3A") (void);
 #endif
//------------------------------------------------
 byte gaby_dcf77_double_buffer[2][DCF_SIZE];
 byte gby_dcf77_flip;
 byte gby_dcf77_puls_count;
 byte gby_dcf77_error;
 flash char gac_parity_error[ERROR_MSG_LENGTH] =
    {'P', 'a', 'r', 'i', 't', 'y', ' ', ' ', ' ', ' ', ' ', 'E', 'r', 'r', 'o', 'r', ' ', ' ', '!', 0x00};
 flash char gac_out_of_range[ERROR_MSG_LENGTH] =
    {'O', 'u', 't', ' ', 'O', 'f', ' ', ' ', ' ', ' ', ' ', 'R', 'a', 'n', 'g', 'e', ' ', ' ', '!', 0x00};
 flash char gac_signal_error[ERROR_MSG_LENGTH] =
    {'S', 'i', 'g', 'n', 'a', 'l', ' ', ' ', ' ', ' ', ' ', 'E', 'r', 'r', 'o', 'r', ' ', ' ', '!', 0x00};
 flash char gac_weekdays[WEEKDAYS][WEEKDAY_LENGTH] =
    {
    ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', 0x00,
    'M', 'o', 'n', 't', 'a', 'g', ' ', ' ', ' ', ' ', 0x00,
    'D', 'i', 'e', 'n', 's', 't', 'a', 'g', ' ', ' ', 0x00,
    'M', 'i', 't', 't', 'w', 'o', 'c', 'h', ' ', ' ', 0x00,
    'D', 'o', 'n', 'n', 'e', 'r', 's', 't', 'a', 'g', 0x00,
    'F', 'r', 'e', 'i', 't', 'a', 'g', ' ', ' ', ' ', 0x00,
    'S', 'o', 'n', 'n', 'a', 'b', 'e', 'n', 'd', ' ', 0x00,
    'S', 'o', 'n', 'n', 't', 'a', 'g', ' ', ' ', ' ', 0x00
    };
//------------------------------------------------
// void DCF77Init(void)
// Initialize the global variables and put the DCF77-databuffer on a defined state.
// Parameter: void
//  No parameters.
// Result: void
//  No returned value.
// Description:
//  Always call this function first.
 void DCF77Init(void)
    {
     int i_index;
// Port D.7 or F.0 = input.
    Port_DataDirBit(DCF_IN, 0);
    Port_DataDirBit(PORT_LED1, 1);
// Pull up on.
    Port_WriteBit(DCF_IN, 1);
    for(i_index = 0; i_index < DCF_SIZE; i_index++)
        {
        gaby_dcf77_double_buffer[0][i_index] = 0;
        gaby_dcf77_double_buffer[1][i_index] = 0;
        }
    gby_dcf77_flip = 0;
    gby_dcf77_puls_count = 0;
    gby_dcf77_error = 0xFF;
 #ifdef MEGA32
    Irq_SetVect(INT_TIM2COMP, DCF77ReceiveSignal);
 #endif
 #ifdef MEGA128
// Use timer 3
    Irq_SetVect(INT_TIM3CMPA, DCF77ReceiveSignal);
    DCF77StartTimer3A();
// 10 ms = 0.01 s
// 1 / 0.01 = 100 Hz
// 14 734 560 MHz / 100 Hz = 147345,6
// (10 ms = 147345,6 cpu-ticks) / pre_8 = (18418,2 count) = 0x47F2
    //Timer_T3Time(0x47F2, PS_8);
 #endif
    Thread_Start(THREAD_DCF77, DCF77ThreadSetSystemTime);
    }
//------------------------------------------------
// void DCF77Destroy(void)
// Stops the evaluation of the dcf77-signal.
// Parameter: void
//  No parameters.
// Result: void
//  No returned value.
// Description:
//  The execution of the timer interrupt function will be stoped.
//  After calling this function you can use the timer for other tasks.
//  The thread 'DCF77ThreadSetSystemTime' will be also terminated.
 void DCF77Destroy(void)
    {
// We give some time to thread 'DCF77ThreadSetSystemTime'.
    Thread_Delay(30);
 #ifdef MEGA32
    Irq_SetVect(INT_TIM2COMP, NULL);
 #endif
 #ifdef MEGA128
    Irq_SetVect(INT_TIM3CMPA, NULL);
 #endif
    Thread_Kill(THREAD_DCF77);
    gby_dcf77_error = 0xFF;
    }
//------------------------------------------------
// BOOL DCF77NotReady(void)
// Get the dcf77-signal status.
// Parameters: void
//  No parameters.
// Result: BOOL
//  Returns true if the dcf77-signal is faulty, otherwise false.
// Description:
//  Use this function to wait for the dcf77-signal.
 BOOL DCF77NotReady(void)
    {
    if(gby_dcf77_error)
        return true;
    return false;
    }
//------------------------------------------------
// void DCF77ReceiveSignal(void)
// This function captures the pulse lengths of the DCF77-signals.
// Parameter: void
//  No parameters.
// Result: void
//  No returned value.
// Description:
//  The system will call this function all 10 ms.
//  When a new minute begins and the pulse is without error, a thread
//  will be resumed to set the system time.
 void DCF77ReceiveSignal(void)
    {
    static byte by_interrupt_count;
    static byte by_status_old;
    static byte by_bit_pos;
    static byte by_byte_count;
    byte by_status_new;
    by_status_new = Port_ReadBit(DCF_IN);
// If flank change?
    if(by_status_new ^ by_status_old)
        {
//------------------------------------------------
// Use the messages to observe the dcf77-signal.
// This is helpfully to adjust the 'case' instructions.
// Afer adapting the 'case' instructions comment out the messages.
// If the dcf77-signal of good quality, only the values:
// 10, 20, 80, 90, 180, 190
// should be displayed!
 #ifdef DEBUG
        Msg_WriteWord(by_interrupt_count);
        Msg_WriteChar(' ');
 #endif
//------------------------------------------------
// If positive flank?
        if(by_status_new)
            {
            Port_ToggleBit(PORT_LED1);
            switch(by_interrupt_count)
                {
/*** To adapt the monitoring of the DCF77-signal change the 'case' instructions. ***/
//------------------------------------------------
// after 200 ms high pulse -> 800 ms low pulse
                //case 75:
                //case 76:
                //case 77:
                //case 78:
                case 79:
                case 80: // <- 800 ms
                case 81:
                //case 82:
                //case 83:
//------------------------------------------------
// after 100 ms high pulse -> 900 ms low pulse
                //case 85:
                //case 86:
                //case 87:
                //case 88:
                case 89:
                case 90: // <- 900 ms
                case 91:
                //case 92:
                //case 93:
//------------------------------------------------
                    gby_dcf77_puls_count++;
                    break;
// Synchronization pulse.
//------------------------------------------------
// after 200 ms high pulse:
// 800 ms low pulse + 1000 ms synchronization low pulse
                //case 175:
                //case 176:
                //case 177:
                //case 178:
                case 179:
                case 180: // <- 1800 ms
                case 181:
                //case 182:
                //case 183:
                //case 184:
//------------------------------------------------
// after 100 ms high pulse:
// 900 ms low pulse + 1000 ms synchronization low pulse
                //case 185:
                //case 186:
                //case 187:
                //case 188:
                case 189:
                case 190: // <- 1900 ms
                case 191:
                //case 192:
                //case 193:
                //case 194:
//------------------------------------------------
                    if((0x0F | gby_dcf77_error) == (by_byte_count | by_bit_pos))
                        {
                        gby_dcf77_flip = gby_dcf77_flip ^ 0x01;
                        gby_dcf77_error = 0x00;
                        Thread_Resume(THREAD_DCF77);
                        }
                    else
                        gby_dcf77_error = 0x0F;
                    by_bit_pos = 0x01;
                    by_byte_count = 0x00;
                    gby_dcf77_puls_count = 0x00;
                    break;
                default:
                    gby_dcf77_error = 0xFF;
                }
            }
// Negative flank.
        else
            {
            switch(by_interrupt_count)
                {
//------------------------------------------------
// Logical 0 = 100 ms high pulse
                //case 7:
                //case 8:
                case 9:
                case 10: // <- 100 ms
                case 11:
                //case 12:
                //case 13:
                //case 14:
                //case 15:
//------------------------------------------------
                    gaby_dcf77_double_buffer[gby_dcf77_flip][by_byte_count] =
                        gaby_dcf77_double_buffer[gby_dcf77_flip][by_byte_count] & ~by_bit_pos;
                    break;
//------------------------------------------------
// Logical 1 = 200 ms high pulse
                //case 17:
                //case 18:
                case 19:
                case 20: // <- 200 ms
                case 21:
                //case 22:
                //case 23:
                //case 24:
                //case 25:
//------------------------------------------------
                    gaby_dcf77_double_buffer[gby_dcf77_flip][by_byte_count] =
                        gaby_dcf77_double_buffer[gby_dcf77_flip][by_byte_count] | by_bit_pos;
                    break;
                default:
                    gby_dcf77_error = 0xFF;
                }
            by_bit_pos = by_bit_pos << 1;
            if(!by_bit_pos)
                {
                by_bit_pos = 0x01;
                by_byte_count = 0x07 & ++by_byte_count;
                }
            }
        by_interrupt_count = 0x00;
        }
    else
// If no pulse?
        if(194 < by_interrupt_count)
            gby_dcf77_error = 0xFF;
    by_status_old = by_status_new;
 #ifdef MEGA32
    by_interrupt_count = by_interrupt_count + Irq_GetCount(INT_TIM2COMP);
 #endif
 #ifdef MEGA128
// Use timer 3.
    by_interrupt_count = by_interrupt_count + Irq_GetCount(INT_TIM3CMPA);
 #endif
    }
//------------------------------------------------
// void DCF77ThreadSetSystemTime(void)
// This function sets the system time.
// Parameter: void
//  No parameter.
// Result: void
//  No returned value.
// Description:
//  The thread is stopped immediately after he is started.
//  If a faultless data acquisition is ended, he is woken up with
//  the first positive flank of the new minute.
//  Since the DCF77-signal is interference-prone, it should be checked before use.
//  It is my opinion, better to have no time as using the wrong time.
//  If you think differently, please feel free to remove some checks.
 void DCF77ThreadSetSystemTime(void)
    {
    byte aby_dcf[DCF_SIZE];
    int ai_values[VALUES_SIZE];
    byte aby_bcd[BCD_SIZE];
    while(1)
        {
        Thread_Wait(THREAD_DCF77, SIG_DCF77);
        if(DCF77CopyBuffer(aby_dcf))
            {
            if(DCF77CheckParity(aby_dcf))
                {
                 DCF77ToBCD(aby_dcf, aby_bcd);
                DCF77BCDToInteger(aby_bcd, ai_values);
                 if(DCF77CheckRange(ai_values))
                    {
                    Clock_SetTime(ai_values[HOUR], ai_values[MINUTE], gby_dcf77_puls_count, 0x00);
                    Clock_SetDate(ai_values[DAY] - 1, ai_values[MONTH] - 1, ai_values[YEAR]);
                    }
                }
            }
        }
    }
//------------------------------------------------
// BOOL DCF77CopyBuffer(byte pby_dcf[])
// This function copies the current DCF77-data in a array.
// Parameter: byte pby_dcf[]
//  Pointer to an array of bytes with the size of 8 bytes (DCF_SIZE),
//  in which the DCF77-data are copied.
// Result: BOOL
//  Is the copy-operation successful  the function returns true. Otherwise false.
// Description:
//  You should not access directly the global array 'g_aby_dcf77_double_buffer'!
//  Firstly: Because the DCF77-data can be faulty.
//  Secondly: Reading and writing to the same array at the same time
//  can cause problems.
 BOOL DCF77CopyBuffer(byte pby_dcf[])
    {
    byte by_flop;
    int i_index;
    if(gby_dcf77_error)
        return false;
    by_flop = gby_dcf77_flip ^ 0x01;
    for(i_index = 0; i_index < DCF_SIZE; i_index++)
        pby_dcf[i_index] = gaby_dcf77_double_buffer[by_flop][i_index];
    return true;
    }
//------------------------------------------------
// BOOL DCF77CheckParity(byte pby_dcf[])
// Checks the parity of the DCF77-Signal for minute, hour and date.
// Parameter: byte pby_dcf[]
//  Pointer to an array which contains the DCF77-data.
// Result: BOOL
//  Is the  check successful  the function returns true. Otherwise false.
// Description:
 BOOL DCF77CheckParity(byte pby_dcf[])
    {
    int i_byte;
    int i_bit;
    int i_pos;
    int i_count;
    i_pos = 0;
    for(i_byte = 2; i_byte <= 7; i_byte++)
        {
        for(i_bit = 1; i_bit <= 128; i_bit = i_bit << 1)
            {
            if(pby_dcf[i_byte] & i_bit)
                i_count++;
            switch(i_pos)
                {
                case 4: i_count = 0; break;
                case 12:
                case 19: if(i_count & 1) return false; break;
                case 42:
                    if(i_count & 1)
                         return false;
                     else
                         return true;
                }
            i_pos++;
            }
        }
    return false;
    }
//------------------------------------------------
// BOOL DCF77CheckRange(int pi_values[])
// This function checks whether the data lie in the allowed field for minute, hour,
// day, weekday, month and year.
// Parameter: int pi_values[]
//  Pointer to an array which contains the DCF77-data as integer values.
// Result: BOOL
//  Is the  check successful  the function returns true. Otherwise false.
// Description:
//  If all data lie in the permissible field true is returned.
//  The value for the year must be adjusted.
 BOOL DCF77CheckRange(int pi_values[])
     {
     if((0 > pi_values[MINUTE]) || (59 < pi_values[MINUTE]))
         return false;
     if((0 > pi_values[HOUR]) || (23 < pi_values[HOUR]))
         return false;
     if((1 > pi_values[DAY]) || (31 < pi_values[DAY]))
         return false;
     if((1 > pi_values[WEEKDAY]) || (7 < pi_values[WEEKDAY]))
         return false;
     if((1 > pi_values[MONTH]) || (12 < pi_values[MONTH]))
         return false;
     if((12 > pi_values[YEAR]) || (20 < pi_values[YEAR]))
         return false;
     return true;
     }
//------------------------------------------------
// void DCF77ToBCD(byte pby_dcf, byte pby_bcd[])
// In this function the DCF77 signal data are moved into a BCD data format.
// 1. Parameter: byte pby_dcf[]
//  Pointer to an array with the size of 8 bytes (DCF_SIZE) which contains
//  the DCF77-data.
// 2. Parameter: byte pby_bcd[]
//  Pointer to an array with the size of 6 bytes (BCD_SIZE) into which
//  the values will be inserted.
// Result: void
//  No returned value.
// Description:
//  The DCF77-data will be inserted in the array in BCD-format.
//  Except of "weekday" every byte contains two decades.
 void DCF77ToBCD(byte pby_dcf[], byte pby_bcd[])
    {
    pby_bcd[MINUTE] = ((pby_dcf[2] >> 5) | (pby_dcf[3] << 3)) & 127;
    pby_bcd[HOUR] = ((pby_dcf[3] >> 5) | (pby_dcf[4] << 3)) & 63;
    pby_bcd[DAY] = ((pby_dcf[4] >> 4) | (pby_dcf[5] << 4)) & 63;
    pby_bcd[WEEKDAY] = (pby_dcf[5] >> 2) & 7;
    pby_bcd[MONTH] = ((pby_dcf[5] >> 5) | (pby_dcf[6] << 3)) & 31;
    pby_bcd[YEAR] = (pby_dcf[6] >> 2) | (pby_dcf[7] << 6);
    }
//------------------------------------------------
// void DCF77BCDToInteger(byte pby_bcd[], int pi_values[])
// This function converts the DCF77-data from BCD-format into the
// corresponding integer values.
// 1. Parameter: byte pby_bcd[]
//  Pointer to an array which contains the DCF77-data in BCD-format.
// 2. Parameter: int pi_values[]
//  Pointer to an array with the size of 6 integers (VALUES_SIZE) into which
//  the values will be inserted.
// Result: void
//  No returned value.
// Description:
//  The integer values are better suitable for the further processing
//  in many possible applications.
 void DCF77BCDToInteger(byte pby_bcd[], int pi_values[])
     {
     int i_index;
     for(i_index = MINUTE; i_index <= YEAR; i_index++)
         pi_values[i_index] = (((pby_bcd[i_index] >> 4) * 10) + (pby_bcd[i_index] & 0x0F));
     }
//------------------------------------------------
// void DCF77BCDToString(byte pby_bcd[], char p_string[])
// This function converts the BCD-coded DCF77-data into a readable string.
// 1. Parameter: byte pby_bcd[]
//  Pointer to an array which contains the DCF77-data in BCD-format.
// 2. Parameter: char p_string[]
//  Pointer to an array with the size of 35 chars (MSG_LENGTH) into which
//  the values will be inserted.
// Result: void
//  No returned value.
// Description:
//  The character string has the format: "06-08-2012 14:13:06 weekday DCF"
 void DCF77BCDToString(byte pby_bcd[], char p_string[])
    {
    int i_index;
    int i_pos;
    p_string[0] = (pby_bcd[DAY] >> 4) + 0x30;
    p_string[1] = (pby_bcd[DAY] & 0x0F) + 0x30;
    p_string[2] = '-';
    p_string[3] = (pby_bcd[MONTH] >> 4) + 0x30;
    p_string[4] = (pby_bcd[MONTH] & 0x0F) + 0x30;
    p_string[5] = '-';
    p_string[6] = '2';
    p_string[7] = '0';
    p_string[8] = (pby_bcd[YEAR] >> 4) + 0x30;
    p_string[9] = (pby_bcd[YEAR] & 0x0F) + 0x30;
    p_string[10] = ' ';
    p_string[11] = (pby_bcd[HOUR] >> 4) + 0x30;
    p_string[12] = (pby_bcd[HOUR] & 0x0F) + 0x30;
    p_string[13] = ':';
    p_string[14] = (pby_bcd[MINUTE] >> 4) + 0x30;
    p_string[15] = (pby_bcd[MINUTE] & 0x0F) + 0x30;
    p_string[16] = ':';
    p_string[17] = ((gby_dcf77_puls_count % 100) / 10) + 0x30;
    p_string[18] = (gby_dcf77_puls_count % 10) + 0x30;
// Because there is no flank at the 59. second, we must simulate the last second of
// the minute.
    if(58 == gby_dcf77_puls_count)
        gby_dcf77_puls_count++;
    p_string[19] = ' ';
    i_index = 20;
    i_pos = 0;
    while(gac_weekdays[pby_bcd[WEEKDAY]][i_pos])
         p_string[i_index++] = gac_weekdays[pby_bcd[WEEKDAY]][i_pos++];
    p_string[i_index++] = ' ';
    p_string[i_index++] = 'D';
    p_string[i_index++] = 'C';
    p_string[i_index++] = 'F';
    p_string[i_index] = 0x00;
    }
//------------------------------------------------
// void DCF77TimeAsString(char p_string[])
// This function copies the DCF77-time into a readable character string.
// Parameter: char p_string[]
//  Pointer to an array of char with the size of 35 (MSG_LENGTH).
// Result: void
//  No returned value.
// Description:
//  If you need the DCF77-time in form of a string, this function does
//  the whole work. If during the checks an error is found, then an
//  error-message is copied into the buffer.
 void DCF77TimeAsString(char p_string[])
    {
    byte aby_dcf[DCF_SIZE];
    int ai_values[VALUES_SIZE];
    byte aby_bcd[BCD_SIZE];
    int i_index;
    i_index = 0;
    if(DCF77CopyBuffer(aby_dcf))
        {
        if(DCF77CheckParity(aby_dcf))
            {
 	    	DCF77ToBCD(aby_dcf, aby_bcd);
	    	DCF77BCDToInteger(aby_bcd, ai_values);
 	    	if(DCF77CheckRange(ai_values))
                DCF77BCDToString(aby_bcd, p_string);
            else
                do
                    p_string[i_index] = gac_out_of_range[i_index];
                    while(p_string[i_index++]);
            }
        else
            do
                p_string[i_index] = gac_parity_error[i_index];
                while(p_string[i_index++]);
        }
    else
        do
            p_string[i_index] = gac_signal_error[i_index];
            while(p_string[i_index++]);
    }
//------------------------------------------------
